//==============================================================================
//
//  Title     : Ni4882.h
//  Copyright : National Instruments 1999.  All Rights Reserved.
//  Purpose   : Declares the following classes:
//
//                  CNi4882Address
//                  CNi4882AddressList
//                  CNi4882
//                  CNi4882Device
//                  CNi4882Board
//                  CNi4882Exception
//
//==============================================================================
#pragma once

//==============================================================================
//  Includes
//==============================================================================
#include <vector>
#include "NiCommon.h"
#include "NiVector.h"
#include "Ni4882EventProxy.h"


//==============================================================================
//  Namespace Directives
//==============================================================================
namespace NI {


//==============================================================================
//  Macros
//==============================================================================
#define On4882EventHandler(className, handler) ((int (className::*)(CNi4882&, void*))handler)


//==============================================================================
//  Type Definitions
//==============================================================================
typedef short Addr4882;                       // 488.2 driver address structure


// A CNi4882Address object encapsulates an IEEE-488.2 address, which 
// consists of a primary address in the range 0 to 30 and an optional secondary 
// address in the range of 96 to 126.
//
class CNi4882Address {
    public:
        //---------------------------------------------------------------------
        //  Static Data
        //---------------------------------------------------------------------

        // Designates that there is no secondary address.
        static const int NoSAD;                                                         
        // Designates all secondary addresses.
        static const int AllSAD;                                                        
        
        //---------------------------------------------------------------------
        //  Constructors/Destructor
        //---------------------------------------------------------------------
        
        // Constructs an address from its component parts.
        CNi4882Address (
            // Primary address in the range of 0 to 30.
            int pad, 
            // Secondary address in the range of 96 to 126. Set to CNi4882Address::NoSAD 
            // to disable secondary addressing.
            int sad = NoSAD);

        // Copy constructor.
        CNi4882Address (
            // Object to copy.
            const CNi4882Address& source);

        // Destructor
        virtual ~CNi4882Address ();                                                     
        
        //---------------------------------------------------------------------        
        //  Operators                                                                   
        //---------------------------------------------------------------------        
        
        // Converts the address object into a format used by the 488.2 driver.
        operator Addr4882 () const;                                                     
        // Assignment operator.
        CNi4882Address& operator = (
            // Object to copy.
            const CNi4882Address& source);

        // Compares two CNi4882Address objects to determine if they are 
        // equal.
        bool operator == (
            // Object against which to compare.
            const CNi4882Address& other
        ) const;                           
        
        //---------------------------------------------------------------------        
        //  Functions                                                                     
        //---------------------------------------------------------------------        
        // Gets the primary address.
        int GetPAD () const;                                                            
        // Gets the secondary address.
        int GetSAD () const;                                                            
        // Gets the entire address.
        Addr4882 GetAddress () const;                                                   
        // Gets the result associated with this address.
        short GetResult () const;                                                       
        // Sets the primary component of the address.
        void SetPAD (
            // Primary address in the range of 0 to 30.
            int pad);

        // Sets the secondary component of the address.
        void SetSAD (
            // Secondary address in the range of 96 to 126. Set to zero
            // to disable secondary addressing.
            int sad);

        // Sets the result associated with this address.
        void SetResult (
            // The associated result value.
            short result);

        // Sets the entire address.
        void SetAddress (
            // Full address, primary, and secondary combined.
            Addr4882 fullAddress);
                                                                                        
    protected:                                                                          
        //---------------------------------------------------------------------        
        //  Functions                                                                     
        //---------------------------------------------------------------------        

        // Copies the specified address to this object.
        void Copy (
            // Object to copy.
            const CNi4882Address& source);
                                                                                        
    private:                                                                            
        //---------------------------------------------------------------------        
        //  Data                                                                        
        //---------------------------------------------------------------------        

        // 488.2 address value.
        union {                                                                         
            // Full address.
            Addr4882 full;                                                              
            struct {                                                                    
                // Primary address component.
                uInt8 pad;                                                              
                // Secondary address component.
                uInt8 sad;                                                              
            };                                                                          
        } address;                                                                      
        // Associated result.
        short result;                                                                   
        
        //---------------------------------------------------------------------
        //  Undefined Canonicals
        //---------------------------------------------------------------------
        
        // Default constructor.
        CNi4882Address ();                                                              
};


// A CNi4882AddressList object contains a list of CNi4882Address 
// objects. Pass a CNi4882AddressList object as a parameter to functions you 
// call on CNi4882Board objects to specify the devices to operate on. Refer 
// to the NI-4882 Function Reference Manual for Windows and the 
// NI-4882 User Manual for more information about 488.2. These manuals are
// available on www.ni.com.
class CNi4882AddressList {
    public:
        //---------------------------------------------------------------------
        //  Constructors\Destructor
        //---------------------------------------------------------------------

        // Default constructor.
        CNi4882AddressList ();                                                          

        // Constructs a list containing a single address.
        CNi4882AddressList (
            // 488.2 address to add to list.
            const CNi4882Address& address);

        // Constructs a list containing multiple addresses.
        CNi4882AddressList (
            // Array containing the addresses to add to the list.
            const CNi4882Address addressList[], 
            // Number of elements in addressList.
            int count);

        // Copy constructor.
        CNi4882AddressList (
            // Object to copy.
            const CNi4882AddressList& source);

        // Destructor
        virtual ~CNi4882AddressList ();

        //---------------------------------------------------------------------        
        //  Operators                                                                   
        //---------------------------------------------------------------------        
        // Returns the address at the specified index.
        // A CNiIndexException is thrown if the index is out of bounds.
        CNi4882Address operator [] (
            // Element index.
            int index
        ) const;
        // Returns a reference to the address at the specified index.
        // A CNiIndexException is thrown if the index is out of bounds.
        CNi4882Address& operator [] (
            // Element index.
            int index);
        // Returns the address at the specified index.
        // A CNiIndexException is thrown if the index is out of bounds.
        CNi4882Address operator [] (
            // Element index.
            unsigned int index
        ) const;
        // Returns a reference to the address at the specified index.
        // A CNiIndexException is thrown if the index is out of bounds.
        CNi4882Address& operator [] (
            // Element index.
            unsigned int index);

        // Assignment operator.
        CNi4882AddressList& operator = (
            // Object to copy.
            const CNi4882AddressList& source);

        //---------------------------------------------------------------------        
        //  Functions                                                                     
        //---------------------------------------------------------------------        

        // Gets the number of entries in the address list.
        int GetSize () const;                                                           
        // Gets the result associated with the specified address.
        short GetResult (
            // Address to query.
            const CNi4882Address& address
        ) const;                          
        // Gets the result associated with the address as the specified index.
        short GetResult (
            // Index of the list entry to query.
            int index = 0
        ) const;                                            
        // Sets the result associated with the address as the specified index.
        void SetResult (
            // Index of the list entry to set.
            int index, 
            // New result value.
            short result);

        // Adds an address to the list.
        void Add (
            // The address object to add to the list.
            const CNi4882Address& address);

        // Removes an address from the list.
        bool Remove (
            // The address object to remove from the list.
            const CNi4882Address& address);


    protected:
        //---------------------------------------------------------------------
        //  Type Definitions
        //---------------------------------------------------------------------

        // Address list iterator.
        typedef std::vector<CNi4882Address>::iterator AddressListIterator;              
        // Address list iterator (const).
        typedef std::vector<CNi4882Address>::const_iterator ConstAddressListIterator;   

        //---------------------------------------------------------------------
        //  Functions
        //---------------------------------------------------------------------

        // Copies from the specified address list.
        void Copy (
            // Address list to copy.
            const CNi4882AddressList& source);

        // Searches for the specified address in the list.
        AddressListIterator Find (
            // Address to find in list.
            const CNi4882Address& address);

        // Searches for the specified address in the list.
        ConstAddressListIterator Find (
            // Address to find in list.
            const CNi4882Address& address
        ) const;            
        
    private:
        //---------------------------------------------------------------------
        //  Data
        //---------------------------------------------------------------------

        // Address list data.
        std::vector<CNi4882Address> addressList;
};

// CNi4882 is the base class for the CNi4882Device and CNi4882Board 
// classes. It provides the following functionality that is common to both 
// devices and interface boards. 
//
// * I/O
// * Configuration
// * Event handling
//
// Notes: 
// * You cannot explicitly instantiate objects of this type.
// * Unless otherwise noted, functions in this class return the current status code.
// When exceptions are enabled and the current status code indicates an error, the
// functions in this class throw exceptions instead of returning the current status code.
// See CNi4882::GetStatus for a list of status codes. 
// 
// Refer to the NI-4882 Function Reference Manual for Windows and the 
// NI-4882 User Manual for more information about 488.2. These manuals are
// available on www.ni.com.
class CNi4882 {
    public:
        //---------------------------------------------------------------------
        //  Type Definitions
        //---------------------------------------------------------------------

        // External event handler prototype.
        typedef int (__cdecl * EventHandler)(CNi4882& object, void* userData);

        //---------------------------------------------------------------------
        //  Enumerated Types
        //---------------------------------------------------------------------

        // I/O operation timeout values.
        enum TimeoutValue {
            // infinite timeout (disabled)
            tNone  = 0,
            // 10 microseconds
            t10us  = 1,
            // 30 microseconds
            t30us  = 2,
            // 100 microseconds
            t100us = 3,
            // 300 microseconds
            t300us = 4,
            // 1 milliseconds
            t1ms   = 5,
            // 3 milliseconds
            t3ms   = 6,
            // 10 milliseconds
            t10ms  = 7,
            // 30 milliseconds
            t30ms  = 8,
            // 100 milliseconds
            t100ms = 9,
            // 300 milliseconds
            t300ms = 10,
            // 1 seconds
            t1s    = 11,
            // 3 seconds
            t3s    = 12,
            // 10 seconds
            t10s   = 13,
            // 30 seconds
            t30s   = 14,
            // 100 seconds
            t100s  = 15,
            // 300 seconds
            t300s  = 16,
            // 1000 seconds
            t1000s = 17,                    
        };

        //---------------------------------------------------------------------
        //  Static Data
        //---------------------------------------------------------------------
            //-----------------------------------------------------------------
            //  Status Flags
            //-----------------------------------------------------------------

            // Error detected.
            static const int StatusError;                                               
            // Indicates that a timeout period expired.
            static const int StatusTimeout;                                             
            // END or EOS detected.
            static const int StatusEnd;                                                 
            // Service Request (SRQ) detected by Controller-In-Charge (CIC).
            static const int StatusSRQ;                                                 
            // Device needs service.
            static const int StatusServiceRequest;                                      
            // I/O completed.
            static const int StatusIoComplete;                                          
            // Local Lockout (LLO) state.
            static const int StatusLocalLockout;                                        
            // Remote state.
            static const int StatusRemoteState;                                         
            // Controller-In-Charge (CIC).
            static const int StatusControllerInCharge;
            // Attention asserted.
            static const int StatusAttention;
            // Talker active.
            static const int StatusTalkerActive;
            // Listener active.
            static const int StatusListenerActive;
            // Device Trigger State (DTS).
            static const int StatusDeviceTriggerState;
            // Device Clear State (DCS).
            static const int StatusDeviceClearState;

            //-----------------------------------------------------------------
            //  Errors
            //-----------------------------------------------------------------

            // System error.
            static const int ErrorSystem;                                               
            // Function requires the 488.2 board to be CIC.
            static const int ErrorBoardNotCIC;                                          
            // Write function detected no Listeners.
            static const int ErrorNoListeners;                                          
            // Interface board not addressed correctly.
            static const int ErrorNotAddressed;                                         
            // Invalid argument to function call.
            static const int ErrorInvalidArgument;                                      
            // Function requires the 488.2 board to be System Controller (SAC).
            static const int ErrorBoardNotSAC;                                          
            // I/O operation aborted.
            static const int ErrorIoAborted;                                            
            // Non-existent interface board.
            static const int ErrorNoBoard;                                              
            // Error performing Direct Memory Access (DMA).
            static const int ErrorDMA;                                                  
            // I/O operation started before previous operation completed.
            static const int ErrorEOIP;                                                 
            // The board or device does not support the operation.
            static const int ErrorOperationUnsupported;                                 
            // File system operation error.
            static const int ErrorFile;                                                 
            // Command error during device call.
            static const int ErrorCommand;                                              
            // Serial poll status byte lost.
            static const int ErrorSerialPollByteLost;                                   
            // SRQ remains asserted.
            static const int ErrorSRQ;                                                  
            // Return buffer is full.
            static const int ErrorBufferFull;                                           
            // Buffer type does not support asynchronous I/O.
            static const int ErrorAsyncBuffer;

            //-----------------------------------------------------------------
            //  Commands
            //-----------------------------------------------------------------

            // Unlisten command.
            static const int CommandUnlisten;                                           
            // Untalk command.
            static const int CommandUntalk;                                             
            // Go to local.
            static const int CommandGoToLocal;                                          
            // Selected device clear.
            static const int CommandSelectedDeviceClear;                                
            // Parallel poll configure.
            static const int CommandParallelPollConfigure;                              
            // Group execute trigger.
            static const int CommandGroupExecuteTrigger;                                
            // Take control.
            static const int CommandTakeControl;                                        
            // Local lock out.
            static const int CommandLocalLockout;                                       
            // Device clear.
            static const int CommandDeviceClear;                                        
            // Parallel poll unconfigure.
            static const int CommandParallelPollUnconfigure;                            
            // Serial poll enable.
            static const int CommandSerialPollEnable;                                   
            // Serial poll disable.
            static const int CommandSerialPollDisable;                                  
            // Parallel poll enable.
            static const int CommandParallelPollEnable;                                 
            // Parallel poll disable.
            static const int CommandParallelPollDisable;                                

            //-----------------------------------------------------------------
            //  Event Flags
            //-----------------------------------------------------------------

            // Use current timeout period to control.
            static const int EventTimeout;                                             
            // END or EOS detected.
            static const int EventEnd;                                                 
            // Service Request (SRQ) is asserted.
            static const int EventSRQ;                                                 
            // Device requested service.
            static const int EventServiceRequest;                                      
            // I/O completed.
            static const int EventIoComplete;                                          
            // 488.2 interface is in Local Lockout (LLO).
            static const int EventLocalLockout;                                        
            // 488.2 interface is in Remote state.
            static const int EventRemoteState;                                         
            // 488.2 interface is Controller-in-Charge (CIC).
            static const int EventControllerInCharge;
            // Attention is asserted.
            static const int EventAttention;
            // 488.2 interface is Talker.
            static const int EventTalkerActive;
            // 488.2 interface is Listener.
            static const int EventListenerActive;
            // 488.2 interface is in Device Trigger State (DTS).
            static const int EventDeviceTriggerState;
            // 488.2 interface is in Device Clear State (DCS).
            static const int EventDeviceClearState;

            //-----------------------------------------------------------------
            //  Miscellaneous
            //-----------------------------------------------------------------

            // The 488.2 driver could not rearm the event trigger for the 
            // installed event handler.
            static const int EventRearmFailed;                                          
            // Pass this value to the SetEOSMode function to configure the
            // driver to compare all eight bits of the EOS byte rather than the 
            // low seven bits.
            static const int EightBitCompare;                                           
            // Pass this value to the SetEOSMode function to configure the
            // driver to assert the EOI line it sends EOS during write 
            // operations.
            static const int SendENDwithEOS;                                            
            // Pass this value to the SetEOSMode function to configure the 
            // driver to terminate read operations when it detects EOS.
            static const int TerminteReadOnEOS;                                         

        //---------------------------------------------------------------------
        //  Constructors/Destructor
        //---------------------------------------------------------------------
        // Copy constructor.
        CNi4882 (
            // Object to copy.
            const CNi4882& source);

        // Destructor
        virtual ~CNi4882 ();                                                            

        //---------------------------------------------------------------------
        //  Operators
        //---------------------------------------------------------------------
        // Gets the ID used by the 488.2 driver to identify this board/device.
        inline operator int () const;
        // Assignment operator.
        CNi4882& operator = (
            // Object to copy.
            const CNi4882& source);

        //---------------------------------------------------------------------
        //  Functions
        //---------------------------------------------------------------------

        // Reads data from a board/device to a file.
        int ReadToFile (
            // Name of file to receive data.
            const CString& fileName);

        // Reads data from a board/device to a buffer.
        int Read (
            // The buffer to receive the data.
            //
            // Note: The buffer must be at least count bytes in length.
            void* buffer, 
            // Number of bytes to read.
            long count, 
            // Pass true to read asynchronously. Pass false to read 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Reads data from a board/device to a buffer.
        int Read (
            // The buffer to receive the data.
            //
            // Note: This function resizes the buffer if its current size is
            // smaller than the specified count.
            CNiVector& buffer, 
            // The number of elements to read.
            //
            // Note: The value refers to the number of elements, not bytes
            // to read. For example, if count = 4 and buffer is a
            // CNiReal64Vector, which has an element size of eight, this 
            // function reads a total of 32 bytes.
            long count, 
            // Pass true to read asynchronously. Pass false to read 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Reads data from a board/device to a buffer.
        int Read (
            // The buffer to receive the data.
            //
            // Note: The size of the buffer determines the amount of data this
            // function reads. The total number of bytes this function reads
            // equals the number of elements in the buffer times the size of 
            // each element. For example, if buffer is a CNiReal64Vector 
            // that contains five elements, this function reads a total of 40 
            // bytes.
            CNiVector& buffer, 
            // Pass true to read asynchronously. Pass false to read 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Reads data from a board/device to a string.
        int Read (
            // The string to receive the data.
            //
            // Note: This function resizes the string to contain the number
            // of bytes specified in count.
            CString& buffer,
            // Number of bytes to read.
            long count);

        // Writes data from a file to a board/device.
        int WriteFromFile (
            // Name of file that contains the data to write.
            const CString& fileName);

        // Writes data from a buffer to a board/device.
        int Write (
            // The data to write.
            const void* buffer, 
            // The number of bytes to write to the device.
            long count, 
            // Pass true to write asynchronously. Pass false to write 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Writes data from a buffer to a board/device.
        int Write (
            // The data to write.
            const CNiVector& buffer, 
            // Pass true to write asynchronously. Pass false to write 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Writes data from a string to a board/device.
        int Write (
            // The data to write.
            const CString& buffer, 
            // Pass true to write asynchronously. Pass false to write 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Sets the End-Of-String (EOS) termination mode or character.
        int SetEOSMode (
            // The EOS mode.
            //
            int mode);

        // Enables or disables assertion of the EOI line at the end of write 
        // operations.
        int SetEOIMode (
            // Pass true to configure this object to assert the EOI line after
            // write operations send the last byte of data. Pass false to 
            // configure this object to not assert the EOI line.
            bool enableEOI = true);

        // Checks for the presence of a listening device at the specified
        // address.
        int CheckForListener (
            // The address of the device to check.
            const CNi4882Address& address, 
            // True if the device is listening, otherwise false.
            bool& foundListener);

        // Device Level
        //
        // Places the device in local mode.
        //
        // Board Level
        //
        // Places the board in local mode if it is not in a lockout state.
        int GoToLocal ();                                                               

        // Places a board/device online and resets all of its configuration
        // parameters to their pre-configured state.
        int Reset ();                                                                  

        // Sets the primary and secondary addresses for a board/device.
        int SetAddress (
            // Board/device address.
            const CNi4882Address& address);

        // Stops an asynchronous I/O operation.
        int Stop ();                                                                    

        // Sets the timeout period for asynchronous I/O operations.
        int SetTimeout (
            // Specifies the timeout period.
            //
            // Pass one of the following CNi4882::Timeout enumeration 
            // values.
            //
            // * CNi4882::tNone - infinite timeout (disabled)
            // * CNi4882::t10us - 10 microseconds
            // * CNi4882::t30us - 30 microseconds
            // * CNi4882::t100us - 100 microseconds
            // * CNi4882::t300us - 300 microseconds
            // * CNi4882::t1ms - 1 milliseconds
            // * CNi4882::t3ms - 3 milliseconds
            // * CNi4882::t10ms - 10 milliseconds
            // * CNi4882::t30ms - 30 milliseconds
            // * CNi4882::t100ms - 100 milliseconds
            // * CNi4882::t300ms - 300 milliseconds
            // * CNi4882::t1s - 1 seconds
            // * CNi4882::t3s - 3 seconds
            // * CNi4882::t10s - 10 seconds
            // * CNi4882::t30s - 30 seconds
            // * CNi4882::t100s - 100 seconds
            // * CNi4882::t300s - 300 seconds
            // * CNi4882::t1000s -  1000 seconds
            TimeoutValue timeout = t10s);

        // Waits for the specified events.
        int Wait (
            // The events for which to wait.
            //
            int mask);

        // Specifies which events cause the OnEvent virtual function to be 
        // called. Refer to the OnEvent documentation for details on 
        // implementing an event handler.
        int InstallEventHandler (
            // The events to handle.
            //
            int mask);

        // Installs an external event handler for the specified events.
        int InstallEventHandler (
            // The events to handle.
            //
            int mask, 
            // The external event handler function of the form.
            //
            //     int __cdecl EventHandler(CNi4882& object, void* userData);
            EventHandler handler, 
            // Optional user data that is passed to the event handler.
            void* userData = NULL);

        // Installs an external class member function event handler for the
        // specified events.
        template <class EventHandlerClass> int InstallEventHandler (
            // Specifies which events to handle.
            //
            int mask,
            EventHandlerClass& handlerObject,
            // The external event handler function of the form
            //
            //     int EventHandlerClass::EventHandler(CNi4882& object, void* userData);
            //
            int (EventHandlerClass::* handler)(CNi4882& object, void* userData),
            // Optional user data that is passed to the event handler.
            void* userData = NULL
        )
        {
            return InstallEventProxy(mask, new CNi4882ClassEventProxy<EventHandlerClass> (this, handlerObject, handler, userData));
        }
        // Removes the currently installed event handler.
        int RemoveEventHandler ();                                                      
        // Gets the current status code.
        inline int GetStatus () const;                                                  
        // Gets the current error code.
        inline int GetError () const;                                                   
        // Gets the number of elements transferred in the previous I/O
        // operation.
        inline long GetCount () const;                                                   

        //---------------------------------------------------------------------
        //  Virtual Functions
        //---------------------------------------------------------------------

        // Override this function in a derived class to handle asynchronous 
        // events.  Call InstallEventHandler to enable asynchronous events.
        //
        virtual int OnEvent ();                                                         


    protected:
        //---------------------------------------------------------------------
        //  Constructors
        //---------------------------------------------------------------------

        // Default constructor.
        CNi4882 ();                                                                     

        // Constructs a board/device associated with a logical name, such as
        // "dev1" or "gpib0".
        CNi4882 (
            // Board/device name.
            const CString& name);

        //---------------------------------------------------------------------
        //  Functions
        //---------------------------------------------------------------------

        // Initializes the board/device object.
        void Init ();                                                                   

        // Copies from the specified board/device.
        void Copy (
            // The object to copy.
            const CNi4882& source);

        // Internal helper function for performing read operations.
        int Read (
            // Buffer to which to read.
            void* buffer, 
            // Number of elements to read.
            long count, 
            // Asynchronous read flag.
            bool async, 
            // Size of each element read.
            long elementSize);

        // Internal helper function for performing write operations.
        int Write (
            // The data to write.
            const void* buffer, 
            // The number of bytes to write to the device.
            long count, 
            // Asynchronous read flag.
            bool async,
            // Size of each element read.
            long elementSize);

        // Checks the status of a 488.2 operation.
        int CheckStatus ();                                                             

        // Checks the status of a 488.2 operation.
        int CheckStatus (long elementSize);

        // Checks to see if the specified buffer can be used in an asynchronous
        // I/O operation.
        int CheckAsyncBuffer (
            // Asynchronous I/O flag.
            bool async, 
            // Buffer to check.
            const CNiVector& buffer);

        //---------------------------------------------------------------------
        //  Data
        //---------------------------------------------------------------------

        // ID used by the 488.2 driver to identify this board/device.
        int id;                                                                         

    private:
        //---------------------------------------------------------------------
        //  Data
        //---------------------------------------------------------------------

        // Status code.
        int status;                                                                     
        // Error code.
        int error;                                                                      
        // Transfer count.
        long count;                                                                      
        // FOR INTERNAL USE ONLY!
        CNi4882EventProxy* eventProxy;                                                  

        //---------------------------------------------------------------------
        //  Functions
        //---------------------------------------------------------------------

        // FOR INTERNAL USE ONLY!
        void ReleaseEventProxy ();                                                      
        // FOR INTERNAL USE ONLY!
        void SetEventProxy (CNi4882EventProxy* proxy);                                  
        // FOR INTERNAL USE ONLY!
        int InstallEventProxy (int mask, CNi4882EventProxy* proxy);                     

        //---------------------------------------------------------------------
        //  Friends
        //---------------------------------------------------------------------
        friend CNi4882EventProxy;

        //---------------------------------------------------------------------
        //  Static Functions
        //---------------------------------------------------------------------

        // FOR INTERNAL USE ONLY!
        static int __stdcall EventThunk (int ud, int status, int error, long count, void* userData);    
};


// The CNi4882Device class adds functionality to CNi4882 that is unique 
// to devices. You can construct device objects as described below.
//
// * Use a logical device name, such a "dev1", that you configured in
// Measurement and Automation Explorer.
// * Provide explicit configuration information, such as address, associated
// interface board, and timeout.
// 
// Note: Unless otherwise noted, functions in this class return the current status code.
// When exceptions are enabled and the current status code indicates an error, the
// functions in this class throw exceptions instead of returning the current status code.
// See CNi4882::GetStatus for a list of status codes. 
//
// Refer to the NI-4882 Function Reference Manual for Windows and the 
// NI-4882 User Manual for more information about 488.2. These manuals are
// available on www.ni.com.
class CNi4882Device : public CNi4882 {
    public:
        //---------------------------------------------------------------------
        //  Enumerated Types
        //---------------------------------------------------------------------

        
        // Device configuration options.
        enum ConfigureOption {
            // Device's access board, which is 
            // valid for 
            // GetConfigurationOption only.
            cBNA            = 0x0200,
            // The EOS character.
            cEosChar        = 0x000F,
            // Use 7- or 8-bit EOS compare.
            cEosCompare     = 0x000E,
            // Terminate reads on EOS.
            cEosRead        = 0x000C,
            // Assert EOI with EOS character.
            cEosWrite       = 0x000D,
            // Assert EOI with last data byte.
            cEOT            = 0x0004,
            // Primary Address.
            cPAD            = 0x0001,
            // Swap bytes during a read 
            // operation.
            cReadAdjust     = 0x0013,
            // Swap bytes during a write 
            // operation.
            cWriteAdjust    = 0x0014,
            // Repeat Addressing.
            cRepeatAddr     = 0x0006,
            // Secondary Address.
            cSAD            = 0x0002,
            // Set the timeout value for serial 
            // polls.
            cSPollTime      = 0x0018,
            // Timeout Value.
            cTimeout        = 0x0003,
            // Enable/disable device 
            // unaddressing.
            cUnaddress      = 0x001B,       
        };

        //---------------------------------------------------------------------
        //  Constructors/Destructor
        //---------------------------------------------------------------------

        // Constructs a device associated with a logical name, such as "dev1".
        CNi4882Device (
            // Device name.
            const CString& name);

        // Constructs a device according to the specified configuration.
        CNi4882Device (
            // The address of the device to control.
            const CNi4882Address& address, 
            // Specifies which 488.2 interface board controls the device.
            int board = 0,                      
            // Specifies the timeout period for asynchronous I/O operations.
            TimeoutValue timeout = t10s, 
            // EOI mode of the device.
            //
            // Refer to CNi4882::SetEOIMode for additional information.
            bool eoiMode = true, 
            // EOS mode of the device.
            //
            // Refer to CNi4882::SetEOSMode for additional information.
            int eosMode = 0);

        // Copy constructor.
        CNi4882Device (
            // Object to copy.
            const CNi4882Device& source);

        // Destructor
        virtual ~CNi4882Device ();                                                      

        //---------------------------------------------------------------------
        //  Operators
        //---------------------------------------------------------------------

        // Assignment operator.
        CNi4882Device& operator = (
            // Object to copy.
            const CNi4882Device& source);

        //--------------------------------------------------------------------- 
        //  Functions
        //---------------------------------------------------------------------

        // Gets the specified configuration option.
        int GetConfigurationOption (
            // Specifies which configuration option to get.
            //
            ConfigureOption option, 
            // Upon return, contains the current value of the selected 
            // configuration item. 
            //
            // Note: See SetConfigurationOption for an explanation of this 
            // value.
            int& value);

        // Sets the specified configuration option.
        int SetConfigurationOption (
            // Specifies which configuration option to set.
            //
            ConfigureOption option, 
            // The new value of the selected configuration, which depends on
            // the value you specify in the option parameter.
            int value);

        // Clears the device by sending the Selected Device Clear (SDC) command
        // to the device.
        int Clear ();                                                                   

        // Triggers the device by sending the Group Execute Trigger (GET)
        // command to the device.
        int Trigger ();                                                                 

        // Serial polls the device.
        int SerialPoll (
            // Serial poll response. If bit 6 (hex 40) is set, the device is
            // requesting service.
            char& response);

    private:
        //---------------------------------------------------------------------
        //  Undefined Canonicals
        //---------------------------------------------------------------------

        // Default constructor.
        CNi4882Device ();                                                               
};


// The CNi4882Board class adds functionality to CNi4882 that is unique 
// to interface boards. It contains a collection of functions that support both
// the IEEE-488.1 and IEEE-488.2 standards. You use the logical board name to
// construct objects of type CNi4882Board. You can obtain the logical board
// name from Measurement and Automation Explorer.
// 
// Note: Unless otherwise noted, functions in this class return the current status code.
// When exceptions are enabled and the current status code indicates an error, the
// functions in this class throw exceptions instead of returning the current status code.
// See CNi4882::GetStatus for a list of status codes. 
// 
// Refer to the NI-4882 Function Reference Manual for Windows and the 
// NI-4882 User Manual for more information about 488.2. These manuals are
// available on www.ni.com.
class CNi4882Board : public CNi4882 {
    public:
        //---------------------------------------------------------------------
        //  Enumerated Types
        //---------------------------------------------------------------------

        // Board configuration options.
        enum ConfigureOption {
            // Enable/disable Auto Serial 
            // Polling.
            cAutoPoll       = 0x0007,
            // Use the CIC Protocol.
            cCicProtocol    = 0x0008,
            // Use DMA for I/O.
            cDMA            = 0x0012,
            // Remove EOS from END bit of IBSTA.
            cEndBitIsNormal = 0x001A,
            // The EOS character.
            cEosChar        = 0x000F,
            // Use 7- or 8-bit EOS compare.
            cEosCompare     = 0x000E,
            // Terminate reads on EOS.
            cEosRead        = 0x000C,
            // Assert EOI with EOS character.
            cEosWrite       = 0x000D,
            // Assert EOI with last data byte.
            cEOT            = 0x0004,
            // Length of cable specified for 
            // high speed timing.
            cHsCableLength  = 0x001F,
            // Set the IST bit.
            cIST            = 0x0020,
            // Primary Address.
            cPAD            = 0x0001,
            // Secondary Address.
            cSAD            = 0x0002,
            // Use Parallel Poll Mode 2.
            cPP2            = 0x0010,
            // Parallel Poll Configure.
            cPPC            = 0x0005,
            // Set the parallel poll length 
            // period.
            cPPollTime      = 0x0019,
            // Swap bytes during a read 
            // operation.
            cReadAdjust     = 0x0013,
            // Swap bytes during a write 
            // operation.
            cWriteAdjust    = 0x0014,
            // Set the RSV byte.
            cRSV            = 0x0021,
            // Board is System Controller.
            cSC             = 0x000A,
            // Enable/disable the sending of 
            // LLO.
            cSendLLO        = 0x0017,
            // Assert SRE on device calls.
            cSRE            = 0x000B,
            // NORMAL, HIGH, or VERY_HIGH 
            // timing.
            cTiming         = 0x0011,
            // Timeout Value.
            cTimeout        = 0x0003,       
        };

        // End-of-transmission modes.
        enum EOTMode {  
            // Do nothing at the end of a transfer.
            eotNone    = 0,
            // Send newline with EOI after a transfer.
            eotNewline = 1,
            // Assert EOI with the last DAB.
            eotEOI     = 2,         
        };
        
        //---------------------------------------------------------------------
        //  Static Data
        //---------------------------------------------------------------------
        
        // Receive: Possible value of the termination parameter.
        static const int StopEnd;                                                       
        // CheckControlLine: EOI mask.
        static const unsigned short MaskEOI;                                            
        // CheckControlLine: ATN mask.
        static const unsigned short MaskATN;                                            
        // CheckControlLine: SRQ mask.
        static const unsigned short MaskSRQ;                                            
        // CheckControlLine: REN mask.
        static const unsigned short MaskREN;                                            
        // CheckControlLine: IFC mask.
        static const unsigned short MaskIFC;                                            
        // CheckControlLine: NRFD mask.
        static const unsigned short MaskNRFD;                                           
        // CheckControlLine: NDAC mask.
        static const unsigned short MaskNDAC;                                           
        // CheckControlLine: DAV mask.
        static const unsigned short MaskDAV;                                            
        
        //---------------------------------------------------------------------
        //  Constructors/Destructor
        //---------------------------------------------------------------------
        
        // Constructs a board associated with a logical name, such as "gpib0".
        CNi4882Board (
            // Board name.
            const CString& boardName);

        // Copy constructor.
        CNi4882Board (
            // Object to copy.
            const CNi4882Board& source);

        // Destructor
        virtual ~CNi4882Board ();                                                       
        
        //---------------------------------------------------------------------
        //  Operators
        //---------------------------------------------------------------------
        
        // Assignment operator.
        CNi4882Board& operator = (
            // Object to copy.
            const CNi4882Board& source);
        
        //---------------------------------------------------------------------
        //  488.1 Functions
        //---------------------------------------------------------------------
        
        // Gets the specified configuration option.
        int GetConfigurationOption (
            // Specifies which configuration option to get.
            //
            // The configuration option can be a  
            // CNi4882Board::ConfigureOption enumeration value.
            ConfigureOption option, 
            // Upon return, contains the current value of the selected 
            // configuration item. 
            //
            // Note: See SetConfigurationOption for an explanation of this 
            // value.
            int& value);

        // Sets the specified configuration option.
        int SetConfigurationOption (
            // Specifies which configuration option to set.
            //
            // The configuration option can be a 
            // CNi4882Board::ConfigureOption enumeration value.
            ConfigureOption option, 
            // The new value of the selected configuration, which depends on
            // the value you specify in the option parameter.
            int value);

        // Sends commands to the board.
        int SendCommand (
            // Contains the commands to send.
            const void* command, 
            // Number of bytes to send.
            long count, 
            // Pass true to send the command asynchronously. Pass false to send the command 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Sends commands to the board.
        int SendCommand (
            // Contains the commands to send.
            const CNiVector& command, 
            // Pass true to send the command asynchronously. Pass false to send the command 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Sends commands to the board.
        int SendCommand (
            // Contains the commands to send.
            const CString& command, 
            // Pass true to send the command asynchronously. Pass false to send the command 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Makes the board the active controller.
        //  by asserting attention (ATN).
        int BecomeActiveController (
            // Pass true to send the command asynchronously. Pass false to send the command 
            // synchronously. When you use this function asynchronously, 
            // it returns without waiting for the operation to complete. When you 
            // use this function synchronously, it returns after the operation
            // completes.
            bool async = false);

        // Enables or disables the DMA mode of the board.
        int SetDMAMode (
            // Pass true to enable DMA.  Pass false to use programmed I/O
            // to transfer data to and from the board.
            bool enableDMA = true);

        // Sets or clears the individual status (IST) bit for parallel polls.
        int SetIST (
            // Pass true to set the IST bit. Pass false to clear the IST bit.
            bool ist = false);

        // Requests system control to enable the board to send IFC and
        // REN messages to devices.
        int RequestSystemControl ();                                                    

        // Releases system control.
        int ReleaseSystemControl ();                                                    

        // Asserts the Interface Clear (IFC) line.
        int InterfaceClear ();                                                          

        // Asserts the Remote Enable (REN) line.
        int SetRemoteEnableLine ();                                                     

        // Unasserts the Remote Enable (REN) line.
        int ClearRemoteEnableLine ();                                                   

        // Requests service.
        int RequestService (
            // The status byte that the 488-2 interface returns when the 
            // Controller-In-Charge (CIC) serial polls the interface.  If 
            // bit 6 (hex 40) is set, the 488.2 interface asserts the 
            // Service Request (SRQ) line to request service from the 
            // Controller.
            int serialPollStatus);

        // Puts the board in standby mode.
        int GoToStandby (
            // Pass true to enable acceptor handshaking or shadow handshaking.
            // The handshaking is performed until END occurs or until ATN is 
            // reasserted by a subsequent call to BecomeActiveController. 
            // With this option, the 488.2 interface can participate in data
            // handshaking as an acceptor without actually reading data.
            // If END is detected, the interface enters a Not Ready For
            // Data (NRFD) handshake holdoff state which results in holdoff 
            // of subsequent transfers. Pass false to disable handshaking
            // and holdoff.
            bool shadowHandshake = false);

        // Gets the status of the control lines.
        int GetControlLines (
            // On return, contains the current state of the control lines.
            // The low-order byte (bits 0 through 7) contains a mask
            // that indicates the capability of the 488.2 interface to 
            // sense the status of each control line. The upper byte (bits 
            // 8 through 15) contains the control line state information.
            // Following is the pattern of each byte.
            //
            //        7     6     5     4     3     2      1      0
            //     +-----+-----+-----+-----+-----+------+------+-----+
            //     | EOI | ATN | SRQ | REN | IFC | NRFD | NDAC | DAV |
            //     +-----+-----+-----+-----+-----+------+------+-----+
            unsigned short& lines);

        // Checks if the specified control lines are raised.
        bool CheckControlLines (
            // Pass the value you obtained from the lines parameter of 
            // the GetControlLines function.
            unsigned short lines, 
            // Specifies which control lines to test.
            //
            // This value can be any combination of the flags shown below.
            //
            // * CNi4882Board::MaskEOI - End
            // * CNi4882Board::MaskATN - attention
            // * CNi4882Board::MaskSRQ - Service Request
            // * CNi4882Board::MaskREN - Remote Enable
            // * CNi4882Board::MaskIFC - Interface Clear
            // * CNi4882Board::MaskNRFD - not Ready For Data
            // * CNi4882Board::MaskNDAC - not Data Accepted
            // * CNi4882Board::MaskDAV - Data Avaiable
            unsigned short mask);
        
        //---------------------------------------------------------------------
        //  488.2 Functions
        //---------------------------------------------------------------------
        
        // Serial polls multiple devices.
        int SerialPoll (
            // On input, contains a list of devices to serial poll. On
            // output, each CNi4882Address object contains the 
            // serial poll response associated with the device at that
            // address.
            CNi4882AddressList& deviceList);

        // Serial polls a single device.
        int SerialPoll (
            // The address of the device to poll.
            const CNi4882Address& device, 
            // The serial poll response.
            short& response);

        // Clears multiple devices.
        int Clear (
            // Contains a list of addresses of devices to clear.
            const CNi4882AddressList& deviceList);

        // Clears a single device.
        int Clear (
            // The address of the device to clear.
            const CNi4882Address& device);

        // Enables local operations for multiple devices.
        int EnableLocal (
            // Contains a list of addresses of devices to enable.
            const CNi4882AddressList& deviceList);

        // Enables local operation for a single device.
        int EnableLocal (
            // The address of the device to enable.
            const CNi4882Address& device);

        // Enables remote operations for multiple devices.
        int EnableRemote (
            // Contains a list of addresses of devices to enable.
            const CNi4882AddressList& deviceList);

        // Enables remote operations for a single device.
        int EnableRemote (
            // The address of the device to enable.
            const CNi4882Address& device);

        // Queries multiple devices to determine if they are listening.
        int FindListeners (
            // On input, contains a list of devices to query. On output,
            // each CNi4882Address object contains the response 
            // associated with the device at that address.
            CNi4882AddressList& deviceList);

        // Queries a single device to determine if it is listening.
        int FindListeners (
            // The address of the device to query.
            CNi4882Address& device,
            // On return, contains true if the specified device is
            // listening. Otherwise, contains false.
            bool &isListening);

        // Determines which device is requesting service.
        int FindRQS (
            // Contains a list of addresses of devices to query.
            CNi4882AddressList& deviceList, 
            // Serial poll response of the device requesting service.
            short& response, 
            // The index into deviceList of the device requesting
            // service.
            int& deviceIndex);

        // Passes control to the specified device.
        int PassControl (
            // The address of the device to which to pass control.
            const CNi4882Address& device);

        // Conducts a parallel poll.
        int ParallelPoll (
            // The parallel poll response.
            short& response);

        // Configures a device to respond to parallel polls.
        int ParallelPollConfigure (
            // The address of the device to configure.
            const CNi4882Address& device, 
            // Specifies on which data line the device responds to 
            // parallel polls. Valid range is 1 to 8.
            int dataLine, 
            // Line sense (0 or 1) of the parallel poll response.
            int lineSense);

        // Unconfigures all devices for parallel polls.
        int ParallelPollUnconfigure ();                                                                            

        // Unconfigures multiple devices for parallel polls.
        int ParallelPollUnconfigure (
            // Contains a list of addresses of devices to unconfigure.
            const CNi4882AddressList& deviceList);

        // Unconfigures a device for parallel polls.
        int ParallelPollUnconfigure (
            // The address of the device to unconfigure.
            const CNi4882Address& device);

        // Reads data from a single device.
        int Receive (
            // The address of the device from which to read.
            const CNi4882Address& device, 
            // The buffer to receive the data.
            //
            // Note: The buffer must be at least count bytes in length.
            void* buffer, 
            // The number of bytes to read.
            long count, 
            // Specifies the data termination mode.
            // 
            // Pass CNi4882Board::StopEnd to configure the receive operation to complete
            // when it receives a byte and the EOI line is asserted. Otherwise, 
            // pass an 8-bit EOS character to configure the receive operation to 
            // complete when it receives the EOS character.
            int termination = StopEnd);

        // Reads data from a single device.
        int Receive (
            // The address of the device from which to read.
            const CNi4882Address& device, 
            // The buffer to receive the data.
            //
            // Note: This function resizes the buffer if its current size
            // is smaller than the specified count.
            CNiVector& buffer, 
            // Specifies the number of elements to receive.
            //
            // Note: The value refers to the number of elements, not bytes
            // to read. For example, if count = 4 and buffer is a
            // CNiReal64Vector, which has an element size of eight, a total 
            // of 32 bytes are read.
            long count, 
            // Specifies the data termination mode.
            // 
            // Pass CNi4882Board::StopEnd to configure the receive operation to complete
            // it receives a byte and the EOI line is asserted. Otherwise, pass 
            // an 8-bit EOS character to configure the receive operation to 
            // complete when it receives the EOS character.
            int termination = StopEnd);

        // Reads data from a single device.
        int Receive (
            // The address of the device from which to read.
            const CNi4882Address& device, 
            // The buffer to receive the data.
            //
            // Note: The size of the buffer determines the amount of data to 
            // receive. The total number of bytes received equals the number
            // of elements in the buffer times the size of each element. For 
            // example, if the buffer is a CNiReal64Vector 
            // with five elements, then it receives a total of 40 bytes since 
            // each element is eight bytes in size.
            CNiVector& buffer, 
            // Specifies the data termination mode.
            // 
            // Pass CNi4882Board::StopEnd to configure the receive operation to complete
            // it receives a byte and the EOI line is asserted. Otherwise, pass 
            // an 8-bit EOS character to configure the receive operation to 
            // complete when it receives the EOS character.
            int termination = StopEnd);

        // Reads data from a device that is already configured to talk.
        int Receive (
            // The buffer to receive the data.
            //
            // Note: This function resizes the buffer if its current size
            // is smaller than the specified count.
            CNiVector& buffer, 
            // Specifies the number of elements to receive.
            //
            // Note: The value refers to the number of elements, not bytes
            // to read. For example, if count = 4 and buffer is a
            // CNiReal64Vector, which has an element size of eight, a total 
            // of 32 bytes are read.
            long count, 
            // Specifies the data termination mode.
            // 
            // Pass CNi4882Board::StopEnd to configure the receive operation to complete
            // it receives a byte and the EOI line is asserted. Otherwise, pass 
            // an 8-bit EOS character to configure the receive operation to 
            // complete when it receives the EOS character.
            int termination = StopEnd);

        // Reads data from a device that is already configured to talk.
        int Receive (
            // The buffer to receive the data.
            //
            // Note: The size of the buffer determines the amount of data to 
            // receive. The total number of bytes received equals the number
            // of elements in the buffer times the size of each element. For 
            // example, if the buffer is a CNiReal64Vector with five 
            // elements, then a total of 40 bytes is received since each element 
            // is eight bytes in size.
            CNiVector& buffer, 
            // Specifies the data termination mode.
            // 
            // Pass CNi4882Board::StopEnd to configure the receive operation to complete
            // it receives a byte and the EOI line is asserted. Otherwise, pass 
            // an 8-bit EOS character to configure the receive operation to 
            // complete when it receives the EOS character.
            int termination = StopEnd);

        // Configures a single device to talk.
        int ConfigureReceive (
            // The address of the device to configure.
            const CNi4882Address& device);

        // Resets and initializes multiple 488.2-compliant devices.
        int ResetSystem (
            // Contains a list of addresses of devices to reset.
            const CNi4882AddressList& deviceList);

        // Resets and initializes a single 488.2-compliant device.
        int ResetSystem (
            // The address of the device to reset.
            const CNi4882Address& device);

        // Writes data to multiple devices.
        int Send (
            // Contains a list of addresses of devices to which to write.
            const CNi4882AddressList& deviceList, 
            // The data to write.
            const void* buffer, 
            // The number of bytes to write.
            long count, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to multiple devices.
        int Send (
            // Contains a list of addresses of devices to which to write.
            const CNi4882AddressList& deviceList, 
            // The data to write.
            const CNiVector& buffer, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to multiple devices.
        int Send (
            // Contains a list of addresses of devices to which to write.
            const CNi4882AddressList& deviceList, 
            // The data to write.
            const CString& buffer, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to a single device.
        int Send (
            // The address of the device to which to write.
            const CNi4882Address& device, 
            // The data to write.
            const void* buffer, 
            // The number of bytes to read.
            int count, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to a single device.
        int Send (
            // The address of the device to which to write the data.
            const CNi4882Address& device, 
            // The data to write.
            const CNiVector& buffer, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to a single device.
        int Send (
            // The address of the device to which to write the data.
            const CNi4882Address& device, 
            // The data to write.
            const CString& buffer, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to all devices that are configured to listen.
        int Send (
            // The data to write.
            //
            // Note: The buffer must be at least count bytes in length.
            const void* buffer, 
            // The number of bytes to read.
            int count, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to all devices that are configured to listen.
        int Send (
            // The data to write.
            const CNiVector& buffer, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Writes data to all devices that are configured to listen.
        int Send (
            // The data to write.
            const CString& buffer, 
            // Specifies the data termination mode, which controls what
            // happens when the last byte of the buffer is transferred
            // to the device.
            //
            // The data termination mode can be one of the values listed below.
            //
            // * __class__::eotEOI - the EOI line asserted.
            // * __class__::eotNewline - a new line character is sent and the EOI line 
            // is asserted.
            // * __class__::eotNone - this function does nothing in addition to sending
            // * the data.
            //
            EOTMode mode = eotEOI);

        // Configures multiple devices to listen.
        int ConfigureSend (
            // Contains a list of addresses of devices to configure.
            const CNi4882AddressList& deviceList);

        // Configures a single device to listen.
        int ConfigureSend (
            // The address of the device to configure.
            const CNi4882Address& device);

        // Sends the Local Lockout (LLO) message to all devices.
        int SendLocalLockout ();                                                                                   

        // Places multiple devices in the Remote With Lockout State (RWLS).
        int SetRemoteWithLockout (
            // Contains a list of addresses of devices to configure.
            const CNi4882AddressList& deviceList);

        // Causes a multiple 488.2-compliant devices to conduct self tests.
        int TestSystem (
            // Contains a list of addresses of devices to self test.
            CNi4882AddressList& deviceList);

        // Causes a single 488.2-compliant device to conduct a self test.
        int TestSystem (
            // The address of the device to self test.
            CNi4882Address& device);

        // Triggers all devices that are addressed to listen.
        int Trigger ();                                                                                            

        // Triggers multiple devices.
        int Trigger (
            // Contains a list of addresses of devices to trigger.
            const CNi4882AddressList& deviceList);

        // Triggers a single device.
        int Trigger (
            // The address of the device to trigger.
            const CNi4882Address& device);

        // Checks the current state of the Service Request (SRQ) line.
        int CheckForSRQ (
            // On return, contains a non-zero value if SRQ line is asserted.
            // Otherwise, it contains zero.
            short& response);

        // Waits for a device to assert the Service Request (SRQ) line or the
        // timeout period to expire.
        int WaitForSRQ (
            // On return, contains a non-zero value if SRQ line is asserted.
            // Otherwise it contains zero.
            short& response);

    protected:
        // Internal helper function for performing send operations.
        int CNi4882Board::Send (
            // The data to send.
            const void* buffer, 
            // Specifies the number of elements to send.
            int count, 
            // Specifies the data termination mode.
            EOTMode mode, 
            // Size of each element sent.
            long elementSize);

        // Internal helper function for performing send operations.
        int CNi4882Board::Send (
            // The address of the device to which to write the data.
            const CNi4882Address& address, 
            // The data to send.
            const void* buffer, 
            // Specifies the number of elements to send.
            int count, 
            // Specifies the data termination mode.
            EOTMode mode, 
            // Size of each element sent.
            long elementSize);

        // Internal helper function for performing send operations.
        int CNi4882Board::Send (
            // Contains a list of addresses of devices to which to write.
            const CNi4882AddressList& addressList, 
            // The data to send.
            const void* buffer, 
            // Specifies the number of elements to send.
            long count, 
            // Specifies the data termination mode.
            EOTMode mode, 
            // Size of each element sent.
            long elementSize);

        // Internal helper function for performing send command operations.
        int CNi4882Board::SendCommand (
            // The command data to send.
            const void* command, 
            // Specifies the number of elements to send.
            long count, 
            // Asynchronous write flag.
            bool async, 
            // Size of each element sent.
            long elementSize);

        // Internal helper function for performing receive operations.
        int Receive (
            // The address of the device from which to read.
            const CNi4882Address& address, 
            // The buffer to receive the data.
            void* buffer, 
            // Specifies the number of elements to receive.
            long count, 
            // Specifies the data termination mode.
            int termination, 
            // Size of each element received.
            long elementSize);

    private:
        //---------------------------------------------------------------------
        //  Data
        //---------------------------------------------------------------------
        int boardId;

        //---------------------------------------------------------------------
        //  Undefined Canonicals
        //---------------------------------------------------------------------
        
        // Default constructor.
        CNi4882Board ();                                                                        
};


// A CNi4882Exception object encapsulates a 488.2 run-time error. It 
// contains the status code, error code, and transfer count of the most recent 
// 488.2 operation.
class CNi4882Exception : public CNiException {
    public:
        //---------------------------------------------------------------------
        //  Constructors/Destructor
        //---------------------------------------------------------------------
        
        // Constructs a 488.2 exception object.
        CNi4882Exception (
            // Associated status code.
            //
            int status,
            // Associated error code.
            //
            int error,
            // The number of elements transferred in the previous I/O operation.
            long count,
            // Pass true if you create this exception object on the heap.  For
            // example, pass true if you called new to create this object.
            // Pass false if you create this exception object on the 
            // stack or as a global variable.  For example, pass false if you 
            // created this object as a local variable.
            bool autoDelete);

        // Destructor
        virtual ~CNi4882Exception ();                                                           
        
        //---------------------------------------------------------------------
        //  Functions
        //---------------------------------------------------------------------
        
        // Gets the status code associated with the exception.
        int GetStatus () const;                                                                 

        // Gets the error code associated with the exception.
        int GetError () const;                                                                  

        // Gets the transfer count associated with the exception.
        long GetCount () const;                                                                  

    private:
        //---------------------------------------------------------------------
        //  Data
        //---------------------------------------------------------------------
        
        // Status code.
        int status;                                                                             
        // Error code.
        int error;                                                                              
        // Transfer count.
        long count;                                                                              
        
        //---------------------------------------------------------------------
        //  Undefined Canonicals
        //---------------------------------------------------------------------
        
        // Default constructor.
        CNi4882Exception ();                                                                    
        // Copy constructor.
        CNi4882Exception (const CNi4882Exception&);                                             
        // Assignment operator.
        CNi4882Exception& operator = (const CNi4882Exception&);                                 
};


//==============================================================================
//  Inline Includes
//==============================================================================
#include "Ni4882.inl"

}  // NI namespace

